<?php

namespace App\Http\Controllers\Dashboard;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use App\Notifications\User\UserNotice;

class ProfileController extends Controller
{
    public function index() {
        // old()
        return view('dashboard.profile.index', [
            'user' => User::find(auth()->id()),
        ]);
    }

    public function save(Request $request) {
        $user = User::find(auth()->id());

        $data = $request->validate([
            'avatar' => 'nullable|image|max:4096',
            'name' => 'required|string|max:255',
            'username' => ['required','string','max:255', 'regex:/^\S*$/', 'alpha_num', Rule::unique('users')->ignore($user->id)],
            'phone_number' => ['required','string', Rule::unique('users')->ignore($user->id)],
            'marital_status' => 'required|string|max:255',
            'date_of_birth' => ['required', 'date', 'before_or_equal:' . now()->subYears(18)->format('Y-m-d')],
            'occupation' => 'required|string|max:255',
            'address' => 'required|max:255',
        ]);
        
        if($request->hasFile('avatar')) {
            $path = '/uploads/'.$request->file('avatar')->store('avatar', 'storage');
            $this->updateAvatar($path, $user);
        }
        
        [$userData, $userDetails] = $this->getUserAndDetails($data);
        $user->update($userData);
        $user->details()->update($userDetails);

        $messages = [
            'Your profile has been successfully updated.',
            'If you made this change, no further action is required. However, if you did not update your profile, please contact our support team immediately for assistance.',
            'Thank you for keeping your information up to date!'
        ];
        $user->notify(new UserNotice(subject: 'Profile Updated.', messages: $messages));
        return back()->with('success', $this->message('Profile updated successfully.'));
    }

    private function updateAvatar($path, User $user) {
        $user->details()->update([
            'avatar' => $path,
        ]);
    }

    private function getUserAndDetails(array $data) {
        $user = [
            'name' => $data['name'],
            'username' => $data['username'],
            'phone_number' => $data['phone_number'],
        ];

        $userDetails = [
            'date_of_birth' => $data['date_of_birth'],
            'address' => $data['address'],
            'marital_status' => $data['marital_status'],
            'occupation' => $data['occupation'],
        ];

        return [$user, $userDetails];
    }
}
