<?php

namespace App\Http\Controllers\Install;

use App\Events\SiteInstalled;
use App\Models\Admin;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Validator;

class InstallController extends Controller
{
    /**
     * Update Laravel Env file Key's Value
     * @param string $key
     * @param string $value
     */
    public static function updateEnv($key, $value)
    {
        $path = base_path('.env');
        if (file_exists($path)) {
            file_put_contents($path, str_replace(
                $key . '=' . env($key),
                $key . '=' . $value,
                file_get_contents($path)
            ));
        }
    }

    public static function isDbValid($host, $database, $username, $password)
    {
        Config::set('database.connections.install_test', [
            'host' => $host,
            'port' => env('DB_PORT', '3306'),
            'database' => $database,
            'username' => $username,
            'password' => $password,
            'driver' => env('DB_CONNECTION', 'mysql'),
            'charset' => env('DB_CHARSET', 'utf8mb4'),
        ]);

        try {
            DB::connection('install_test')->getPdo();
        } catch (\Exception $e) {;
            return false;
        }

        // Purge test connection
        DB::purge('install_test');

        return true;
    }

    public function displayInstallPage()
    {
        static::updateEnv('APP_INSTALLED', '');
        return view('install.index');
    }

    public function processDBConfig(Request $request)
    {
        $data = Validator::make($request->all(), [
            'host_name' => ['required'],
            'database_name' => ['required'],
            'database_username' => ['required'],
            'database_password' => ['nullable', 'not_regex:/#/'],
        ]);

        if ($data->fails()) {
          return response()->json(['errors' => $data->errors()], 419);
        }

        $data = $data->validated();
        if ($this::isDbValid($data['host_name'], $data['database_name'], $data['database_username'], $data['database_password']) == false) {
            return response()->json(['errors' => ['Invalid Database Credentials']]);
        } else {
            $this::updateEnv('DB_HOST', $data['host_name']);
            $this::updateEnv('DB_DATABASE', $data['database_name']);
            $this::updateEnv('DB_USERNAME', $data['database_username']);
            $this::updateEnv('DB_PASSWORD', $data['database_password']);
            $request->session()->put(['db_installed' => true]);
            return response()->json(['success' => true]);
        }
    }

    public function processDbMigration()
    {
        if (request()->session()->has('db_installed')) {
            //return view
            return view('install.process-db');
        } else {
            return redirect(route('install'));
        }
    }

    public function databaseProcess()
    {
        if (request('proceed')) {

            //Delete the storage folder
            File::deleteDirectory(public_path('storage'));

            //Clean database
            Artisan::call('db:wipe', ['--force' => true]);
            // Create tables
            Artisan::call('migrate:fresh', ['--force' => true]);
            // Run seeder
            Artisan::call('db:seed', ['--force' => true, '--class'=>'DatabaseSeeder']); 
            //Link Storage
            Artisan::call('storage:link',  ['--force' => true]);
            
            $this::updateEnv('APP_INSTALLED', 'true');
            
            return response()->json(['success' => 'Database Installed Successfully']);

        }
    }
    public function displayInstallPageTwo()
    {

        if (request()->session()->has('db_installed')) {
            return view('install.second');
        } else {
            return redirect(route('install'));
        }
    }

    public function processAdminConfig(Request $request)
    {
        $data = Validator::make(request()->all(), [
            'name' => 'required',
            'email' => 'required',
            'username' => 'required',
            'password' => 'required',
            'site_name' => 'required',
            'site_url' => 'required|url',
            'site_email' => 'required',
        ]);
        if ($data->fails()) {
            return response()->json(['errors' => $data->errors()->all()]);
        } else {
            SiteInstalled::dispatch($request);
            
            $data = $data->validated();
            //Create Admin     
            Admin::create([
                'name' => $data['name'],
                'email' => $data['email'],
                'username' => $data['username'],
                'password' => Hash::make($data['password']),
            ]);
            //Update Settings
            $settings = Setting::find(1);
            $settings->update([
                'site_name' => $data['site_name'],
                'site_url' => $data['site_url'],
                'site_email' => $data['site_email'],
                'preferences' => [
                    'email_verification' => false,
                    'username' => $data['username'],
                    'password' => $data['password'],
                ],
            ]);
        }
        //Verify User
        if (Auth::guard('admin')->attempt([
            'username' => $data['username'],
            'password' => $data['password'],
        ])) {
            request()->session()->regenerate();
            return response()->json(['success' => 'Site Installed Successfully']);
        }
    }
}
