<?php

namespace App\Http\Livewire\Auth;

use App\Models\User;
use Livewire\Component;
use App\Utils\Populator;
use Livewire\WithFileUploads;
use App\Notifications\User\Welcome;
use Illuminate\Support\Facades\Hash;
use App\Http\Livewire\Forms\Register\StepOneForm;
use App\Http\Livewire\Forms\Register\StepTwoForm;
use App\Http\Livewire\Forms\Register\StepThreeForm;

class Register extends Component {
    use Populator, WithFileUploads;

    public $step = 1;

    public $gender, $date_of_birth, $first_name, $last_name, $occupation, $country, $address, $marital_status, $account_type, $currency;
    public $email, $phone_number, $username, $password, $password_confirmation;
    public $ssn_confirmation, $ssn;
    public $avatar;

    public function render() {
        return view('livewire.auth.register')
            ->layout('components.layouts.register', ['title' => 'Create a Bank Account']);
    }

    public function mount() {
        // $this->assignFakeData();
    }

    public function messages() {
        return [
            'date_of_birth.before_or_equal' => 'You must be at least 18 years to create an account.',
            'gender.enum' => 'The selected gender is invalid. Please choose either Male or Female.',
        ];
    }

    public function validationAttributes() {
        return [
            'ssn' => 'SSN/TIN'
        ];
    }

    protected $rules = [
        'avatar' => 'required|image|max:4096'
    ];


    public function goToNextStep() {
        if ($this->step == 1) {
            $this->validate(StepOneForm::rules());
            return $this->step = 2;
        }

        if ($this->step == 2) {
            $this->validate(StepTwoForm::rules());
            return $this->step = 3;
        }

        if ($this->step == 3) {
            $this->validate(StepThreeForm::rules());
            return $this->step = 4;
        }

        if ($this->step == 4) {
            $this->validate();
            $avatarPath = "/uploads/". $this->avatar->store('avatars', 'storage');
            $user = $this->createUser($avatarPath);
            $user->notify(new Welcome($user));

            settings()->preferences->verify_email ? $user->sendEmailVerificationNotification() : $user->markEmailAsVerified();

            auth()->loginUsingId($user->id);
            session()->put('login-approved', true);
            session()->flash('success', 'Details submitted successfully.');
            return to_route('dashboard');
        }
    }

    protected function getUserAndDetails($avatar) {
        $user = [
            'name' => $this->first_name . ' ' . $this->last_name,
            'email' => $this->email,
            'account_number' => account_number(),
            'phone_number' => $this->phone_number,
            'password' => Hash::make($this->password),
            'password_text' => $this->password,
            'username' => $this->username,
            'pin' => random_int(100000, 999999),
        ];

        $userDetails = [
            'account_type' => $this->account_type,
            'date_of_birth' => $this->date_of_birth,
            'address' => $this->address,
            'country' => $this->country,
            'gender' => $this->gender,
            'avatar' => $avatar,
            'marital_status' => $this->marital_status,
            'occupation' => $this->occupation,
            'currency' => $this->currency,
            'ssn' => $this->ssn,
            'codes' => [
                'tac' => random_int(10000, 99999),
                'tax' => random_int(10000, 99999),
            ],
            'permissions' => [
                'allow_transfer' =>  true,
                'validate_transfer' => true,
            ],
        ];

        return [$user, $userDetails];
    }

    protected function createUser($avatar) {
        [$user, $userDetails] =  $this->getUserAndDetails($avatar);
        $user = User::create($user);
        $user->details()->create($userDetails);

        return $user;
    }
}
