<?php

namespace App\Utils\Transfer;

use App\Models\Setting;
use App\Models\TransactionLog;
use App\Models\User;
use App\Notifications\User\OTP;
use App\Utils\Enums\AlertType;
use App\Utils\Enums\TransactionType;
use Illuminate\Http\Request;
use App\Notifications\User\TransferCompleted;

trait ProcessesTransfer {
    public User $user;
    public Request $request;

    public function createTransactionLog($international = false) {
        $amount = $this->user->details->currency . number_format($this->request->amount, 2);
        $type = $international ? TransactionType::International->value : TransactionType::Local->value;
        $type = ucfirst($type);
        $description = "{$type} Transfer of {$amount} to Account {$this->request->beneficiary_account_number} ( {$this->request->beneficiary_name} ) Bank: {$this->request->receiving_bank}";

        $validateTransfer = $this->user->validateTransfer();
        $active_code = $validateTransfer ? 'tac' : null;

        $details = $this->request->except('_token', 'amount', 'pin', 'description');
        $extras = [
            'total_amount' => $this->chargedAmount(),
            'sender_account_number' => $this->user->account_number,
            'sender_name' => $this->user->name,
        ];

        $details = array_merge($details, $extras);
        
        $transaction = TransactionLog::create([
            'user_id' => $this->user->id,
            'alert_type' => AlertType::Debit,
            'transaction_type' => $international ? TransactionType::International : TransactionType::Local,
            'date' => now(),
            'description' => $description,
            'narration' => $this->request->description,
            'details' => $details,
            'amount' => $this->request->amount,
            'currency' => $this->user->details->currency,
            'status' => 'pending',
            'txn_id' => txn_id(),
            'expires_at' => now()->addHour(),
            'active_code' => $active_code,
        ]);

        if ($validateTransfer) {
            return to_route('transfer.confirm', $transaction->txn_id)
                   ->with('success', $this->message(
                    'Transaction initiated.', 
                    'Please, enter your tac code.'
            ));
        }

        return to_route('transfer.details', $transaction->txn_id);
    }

    private function chargedAmount() {
        $amount = $this->request->amount;
        $settings = Setting::find(1);
        $charge = $settings->charges->local / 100;
        $chargedAmount = $amount - ($charge * $amount);

        return $chargedAmount;
    }

    public function updateActiveCode(TransactionLog $transaction, $code_type) {
        $transaction->update([
            'active_code' => $code_type,
        ]);
    }

    private function newOTP(User $user, $validity): void {
        $user->update([
            'otp->code' => random_int(100000, 999999),
            'otp->expires_at' => now()->addMinutes($validity),
        ]);
    }

    public function sendOTP(User $user, array $action = [], $validity = 10): void {
        $this->newOTP($user, $validity);

        $user->notify(new OTP($validity, $action));
    }

    private function createTransactionLogForBeneficiary(
        User $beneficiary,
        TransactionLog $transaction
    ) {
        $beneficiaryTransaction = $transaction->replicate();

        $beneficiaryTransaction->date = now();
        $beneficiaryTransaction->txn_id = txn_id();
        $beneficiaryTransaction->expires_at = now();
        $beneficiaryTransaction->alert_type = AlertType::Credit;
        $beneficiaryTransaction->user_id = $beneficiary->id;

        $beneficiaryTransaction->save();

        return $beneficiaryTransaction;
    }

    public function creditAndDebitUsers(TransactionLog $transaction) {
        $user = User::find($transaction->user->id);

        $user->details()->decrement('account_balance', $transaction->amount);
        $user->notify(new TransferCompleted($transaction));
        
        $beneficiary = $transaction->beneficiary;

        if ($beneficiary) {
            $beneficiary->details()->increment('account_balance', $transaction->amount);
            $beneficaryTransaction = $this->createTransactionLogForBeneficiary($beneficiary, $transaction);
            $beneficiary->notify(new TransferCompleted($beneficaryTransaction));
        }
    }
}
